const express = require('express');
const cors = require('cors');
const { GoogleGenerativeAI } = require('@google/generative-ai');
const fs = require('fs').promises;
const path = require('path');

const app = express();
const port = process.env.PORT || 3000;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static('public')); // Serve static files if needed

// Initialize Gemini AI
const API_KEY = 'AIzaSyCnh3zGu8QtPDbaRrn9oTvxl2KFoh5eRxg';
const genAI = new GoogleGenerativeAI(API_KEY);

// Load company data
let companyData = '';

async function loadCompanyData() {
  try {
    const companyDataPath = path.join(__dirname, 'company-data.txt');
    companyData = await fs.readFile(companyDataPath, 'utf8');
    console.log('Company data loaded successfully');
  } catch (error) {
    console.error('Error loading company data:', error);
  }
}

// Initialize company data on startup
async function startServer() {
  await loadCompanyData();
  
  app.listen(port, () => {
    console.log(`🚀 Altea Chatbot API server running on port ${port}`);
    console.log(`📊 Health check: http://localhost:${port}/api/health`);
    console.log(`💬 Chatbot endpoint: http://localhost:${port}/api/chat`);
    
    // Log environment status
    console.log(`🔑 API Key configured: ${!!API_KEY}`);
    console.log(`📄 Company data loaded: ${!!companyData && companyData.length > 0}`);
  });
}

startServer();
app.post('/api/chat', async (req, res) => {
  try {
    const { message, language = 'en' } = req.body;

    if (!message || !message.trim()) {
      return res.status(400).json({ 
        error: 'Message is required',
        response: 'Please provide a message.'
      });
    }

    if (!API_KEY) {
      return res.status(500).json({ 
        error: 'API key not configured',
        response: 'Sorry, the chatbot is currently unavailable. Please try again later.'
      });
    }

    // Language configuration
    const languageConfig = {
      'en': {
        name: 'English',
        instruction: 'Please respond in English.',
        fallback: 'I can provide information specifically about Altea Packaging.'
      },
      'de': {
        name: 'German (Deutsch)',
        instruction: 'Bitte antworte auf Deutsch.',
        fallback: 'Ich kann spezifische Informationen über Altea Packaging bereitstellen.'
      },
      'fr': {
        name: 'French (Français)', 
        instruction: 'Veuillez répondre en français.',
        fallback: 'Je peux fournir des informations spécifiques sur Altea Packaging.'
      },
      'es': {
        name: 'Spanish (Español)',
        instruction: 'Por favor responde en español.',
        fallback: 'Puedo proporcionar información específica sobre Altea Packaging.'
      },
      'ar': {
        name: 'Arabic (العربية)',
        instruction: 'يرجى الرد باللغة العربية.',
        fallback: 'يمكنني تقديم معلومات محددة حول Altea Packaging.'
      },
      'it': {
        name: 'Italian (Italiano)',
        instruction: 'Per favore rispondi in italiano.',
        fallback: 'Posso fornire informazioni specifiche su Altea Packaging.'
      },
      'pt': {
        name: 'Portuguese (Português)',
        instruction: 'Por favor responda em português.',
        fallback: 'Posso fornecer informações específicas sobre a Altea Packaging.'
      },
      'ru': {
        name: 'Russian (Русский)',
        instruction: 'Пожалуйста, отвечайте на русском языке.',
        fallback: 'Я могу предоставить конкретную информацию об Altea Packaging.'
      },
      'tr': {
        name: 'Turkish (Türkçe)',
        instruction: 'Lütfen Türkçe yanıtlayın.',
        fallback: 'Altea Packaging hakkında özel bilgiler sağlayabilirim.'
      },
      'zh': {
        name: 'Chinese (中文)',
        instruction: '请用中文回答。',
        fallback: '我可以提供关于Altea Packaging的具体信息。'
      }
    };

    const currentLang = languageConfig[language] || languageConfig['en'];

    // Create the model and chat session
    const model = genAI.getGenerativeModel({ model: 'gemini-1.5-flash' });
    
    const chat = model.startChat({
      generationConfig: {
        maxOutputTokens: 800,
      },
    });

    // Create the enhanced prompt with automatic language detection
    const enhancedPrompt = `You are a helpful AI assistant for Altea Packaging Group, a leading flexible packaging converter in the MENA region. 

IMPORTANT: Always respond in the SAME LANGUAGE that the user writes their message in. If they write in German, respond in German. If they write in English, respond in English. If they write in Arabic, respond in Arabic, etc. Automatically detect the user's language and match it exactly.

Be professional, helpful, and accurate. Always maintain a friendly and informative tone.

If asked about something not covered in the company data, politely explain that you can provide information specifically about Altea Packaging (in whatever language the user used).

Company Information:
${companyData}

---

User Question: ${message.trim()}

Please provide a helpful and informative response based on the company information above. Remember to automatically detect the user's language and respond in the EXACT SAME LANGUAGE they used.`;

    // Send message to Gemini
    const result = await chat.sendMessage(enhancedPrompt);
    const response = await result.response;
    const text = response.text();

    res.json({
      response: text,
      timestamp: new Date().toISOString()
    });

  } catch (error) {
    console.error('Chatbot error:', error);
    
    // Handle specific API errors
    if (error.message && error.message.includes('API key not valid')) {
      return res.status(500).json({
        error: 'Invalid API key',
        response: 'Sorry, there\'s a configuration issue. Please contact support.'
      });
    }

    res.status(500).json({
      error: 'Internal server error',
      response: 'Sorry, I encountered an error while processing your request. Please try again in a moment.'
    });
  }
});

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({ 
    status: 'OK', 
    timestamp: new Date().toISOString(),
    companyDataLoaded: !!companyData
  });
});

// Get available languages endpoint
app.get('/api/languages', (req, res) => {
  const languages = {
    'en': { name: 'English', flag: '🇺🇸' },
    'de': { name: 'Deutsch', flag: '🇩🇪' },
    'fr': { name: 'Français', flag: '🇫🇷' },
    'es': { name: 'Español', flag: '🇪🇸' },
    'ar': { name: 'العربية', flag: '🇸🇦' },
    'it': { name: 'Italiano', flag: '🇮🇹' },
    'pt': { name: 'Português', flag: '🇵🇹' },
    'ru': { name: 'Русский', flag: '🇷🇺' },
    'tr': { name: 'Türkçe', flag: '🇹🇷' },
    'zh': { name: '中文', flag: '🇨🇳' }
  };
  
  res.json({
    languages: languages,
    defaultLanguage: 'en',
    total: Object.keys(languages).length
  });
});

// Get company info endpoint (optional)
app.get('/api/company-info', (req, res) => {
  res.json({
    name: 'Altea Packaging Group',
    description: 'Leading flexible packaging converter in the MENA region',
    dataLoaded: !!companyData
  });
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Server error:', error);
  res.status(500).json({
    error: 'Internal server error',
    response: 'Sorry, something went wrong. Please try again later.'
  });
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({
    error: 'Not found',
    response: 'The requested endpoint was not found.'
  });
});

module.exports = app;
