# Altea Packaging Chatbot - Angular Integration Guide

This guide will help you integrate the Altea Packaging AI chatbot into your Angular website.

## 📋 Prerequisites

- Angular 12+ project
- Node.js 16+ for the API server
- Google Gemini API key

## 🚀 Quick Setup

### 1. API Server Setup

First, set up the backend API server that will handle chatbot requests:

```bash
# Navigate to your chatbot directory
cd company-chatbot

# Install API dependencies
npm install express cors @google/generative-ai

# Set your API key
# On Windows:
set API_KEY=your_gemini_api_key_here
# On Mac/Linux:
export API_KEY=your_gemini_api_key_here

# Start the API server
node chatbot-api.js
```

The API server will run on `http://localhost:3001`

### 2. Angular Integration

#### Step 1: Copy Component Files

Copy these files to your Angular project:

```
src/app/chatbot/
├── chatbot.component.ts
├── chatbot.component.html
├── chatbot.component.scss
├── chatbot.service.ts
└── chatbot.module.ts
```

#### Step 2: Update Angular App Module

Add the ChatbotModule to your main app module:

```typescript
// app.module.ts
import { NgModule } from '@angular/core';
import { BrowserModule } from '@angular/platform-browser';
import { HttpClientModule } from '@angular/common/http';

import { AppComponent } from './app.component';
import { ChatbotModule } from './chatbot/chatbot.module';

@NgModule({
  declarations: [
    AppComponent
  ],
  imports: [
    BrowserModule,
    HttpClientModule,
    ChatbotModule // Add this
  ],
  providers: [],
  bootstrap: [AppComponent]
})
export class AppModule { }
```

#### Step 3: Add Chatbot to Your Template

Add the chatbot component to any page where you want it to appear:

```html
<!-- In your main layout component -->
<app-chatbot></app-chatbot>
```

#### Step 4: Configure API Endpoint

Update the chatbot service to point to your API server:

```typescript
// chatbot.service.ts
@Injectable({
  providedIn: 'root'
})
export class ChatbotService {
  private apiUrl = 'http://localhost:3001/api/chatbot'; // Update this URL

  // ... rest of the service
}
```

#### Step 5: Add Assets

Add the Altea logo to your assets folder:
- `src/assets/images/altea-logo-icon.png`

## 🎨 Customization

### Brand Colors

The chatbot uses Altea's brand colors defined in the SCSS file:

```scss
$altea-primary: #2c5aa0;      // Altea blue
$altea-secondary: #1a4480;    // Darker blue
$altea-accent: #f39c12;       // Orange accent
```

### Styling

You can customize the chatbot appearance by modifying `chatbot.component.scss`:

- **Position**: Change `bottom` and `right` values for different positioning
- **Size**: Modify `width` and `height` for different dimensions
- **Colors**: Update the color variables at the top of the SCSS file
- **Animation**: Adjust transition durations and easing functions

### Messages

Customize the welcome message in the component:

```typescript
ngOnInit() {
  this.addMessage(
    "Your custom welcome message here!",
    false
  );
}
```

## 🔧 Configuration Options

### Environment Variables

For production, set up environment variables:

```typescript
// environment.ts
export const environment = {
  production: false,
  chatbotApiUrl: 'http://localhost:3001/api/chatbot'
};

// environment.prod.ts
export const environment = {
  production: true,
  chatbotApiUrl: 'https://your-production-api.com/api/chatbot'
};
```

### API Configuration

Update the service to use environment variables:

```typescript
import { environment } from '../../environments/environment';

@Injectable({
  providedIn: 'root'
})
export class ChatbotService {
  private apiUrl = environment.chatbotApiUrl;
  // ... rest of the service
}
```

## 📱 Mobile Optimization

The chatbot is fully responsive and includes:

- Full-screen overlay on mobile devices
- Touch-friendly interface
- Optimized input handling
- Responsive design breakpoints

## 🔒 Security Considerations

1. **API Key**: Never expose your Google API key in the frontend
2. **CORS**: Configure CORS properly for your domain
3. **Rate Limiting**: Consider implementing rate limiting on the API
4. **Input Validation**: The API includes input validation

## 🚀 Production Deployment

### Backend Deployment

1. Deploy the `chatbot-api.js` to your preferred hosting service
2. Set the `API_KEY` environment variable
3. Update CORS settings for your domain

### Frontend Build

```bash
ng build --prod
```

Update the API URL in production environment file.

## 🧪 Testing

Test the chatbot integration:

1. Start the API server: `node chatbot-api.js`
2. Start your Angular app: `ng serve`
3. Open the application and test the chatbot functionality

## 📊 Monitoring

Monitor the chatbot with these endpoints:

- Health check: `GET /api/health`
- Company info: `GET /api/company-info`

## 🛠 Troubleshooting

### Common Issues

1. **CORS Errors**: Ensure the API server has CORS enabled for your domain
2. **API Key Issues**: Verify the Google API key is set correctly
3. **Module Errors**: Ensure all Angular modules are imported
4. **Styling Issues**: Check that SCSS is compiled correctly

### Debug Mode

Enable console logging in the component for debugging:

```typescript
async sendMessage() {
  console.log('Sending message:', userMessage);
  // ... rest of the method
}
```

## 📞 Support

For issues or questions regarding the chatbot integration, contact:

- **Technical Support**: Marqenti (marqenti.tn)
- **Altea Packaging**: contact@altea-packaging.com

## 🔄 Updates

To update the company data:

1. Update the `company-data.txt` file
2. Restart the API server
3. The chatbot will automatically use the new data

---

**Developed by Marqenti for Altea Packaging**  
This chatbot represents the latest in AI-powered customer service technology, designed to provide accurate and helpful information about Altea Packaging's products and services.
